<?php

defined('BASEPATH') || exit('No direct script access allowed');

require_once __DIR__ . '/../WhatsbotapiController.php';

use Whatsbotapi\WhatsbotapiController;

class Messagebots extends WhatsbotapiController
{
    public function __construct()
    {
        parent::__construct();

        $this->load->model('whatsbotapi_model');
        $this->load->model('bots_model');
        $this->load->library('app_modules');
        $this->load->helper(['whatsbotapi', 'whatsbot/whatsbot']);

        if ($this->app_modules->is_inactive('whatsbot')) {
            $this->response(isWhatsbotNotActive()['response'], isWhatsbotNotActive()['response_code']);
        }

        register_language_files('whatsbotapi');
        load_client_language();

        if (!defined('WHATSBOT_MODULE_UPLOAD_FOLDER')) {
            define('WHATSBOT_MODULE_UPLOAD_FOLDER', 'uploads/whatsbot');
        }

        if (isWbAuthorized()) {
            $this->response(isWbAuthorized()['response'], isWbAuthorized()['response_code']);
        }
    }
    /**
     * @api {get} /whatsbotapi/v1/messagebots List All Message Bots
     *
     * @apiVersion 1.0.0
     *
     * @apiName Get All Message Bots
     *
     * @apiGroup Message Bots
     *
     * @apiHeader {String} Authorization <span class="btn btn-xs btn-danger">Required</span> Basic Access Authentication token.
     *
     * @apiSampleRequest /whatsbotapi/v1/messagebots
     *
     * @apiSuccess {Boolean} status  Response status.
     * @apiSuccess {Array}   data    Message bots.
     * @apiSuccess {String}  message Success message.
     *
     * @apiSuccessExample Success-Response:
     *     HTTP/1.1 200 OK
     *     {
     *          "status": true,
     *          "data": [
     *              {
     *                  "id": "1",
     *                  "name": "test bot1",
     *                  "rel_type": "leads",
     *                  "reply_text": "hello from test bot 1",
     *                  "reply_type": "1",
     *                  "trigger": "testbot",
     *                  "bot_header": "",
     *                  "bot_footer": "",
     *                  "button1": "",
     *                  "button1_id": "",
     *                  "button2": "",
     *                  "button2_id": "",
     *                  "button3": "",
     *                  "button3_id": "",
     *                  "button_name": "",
     *                  "button_url": "",
     *                  "filename": null,
     *                  "addedfrom": "1",
     *                  "created_at": "2024-09-21 12:09:30",
     *                  "is_bot_active": "1",
     *                  "sending_count": "0"
     *              },
     *              {
     *                  "id": "2",
     *                  "name": "bot test2",
     *                  "rel_type": "leads",
     *                  "reply_text": "from bot test 2",
     *                  "reply_type": "1",
     *                  "trigger": "bot2",
     *                  "bot_header": "",
     *                  "bot_footer": "",
     *                  "button1": "",
     *                  "button1_id": "",
     *                  "button2": "",
     *                  "button2_id": "",
     *                  "button3": "",
     *                  "button3_id": "",
     *                  "button_name": "",
     *                  "button_url": "",
     *                  "filename": null,
     *                  "addedfrom": "1",
     *                  "created_at": "2024-09-21 12:34:37",
     *                  "is_bot_active": "1",
     *                  "sending_count": "0"
     *              }
     *          ],
     *          "message": "Data retrieved successfully"
     *      }
     *
     * @apiError {Boolean} status  Response status.
     * @apiError {String}  message No data found.
     *
     * @apiErrorExample Error-Response:
     *     HTTP/1.1 404 Not Found
     *     {
     *       "status": false,
     *       "message": "No data found"
     *     }
     */

    /**
     * @api {get} /whatsbotapi/v1/messagebots/id/:id Request Message Bot information
     *
     * @apiVersion 1.0.0
     *
     * @apiName Request Message Bot information
     *
     * @apiGroup Message Bots
     *
     * @apiHeader {String} Authorization <span class="btn btn-xs btn-danger">Required</span> Basic Access Authentication token.
     *
     * @apiParam {Number} id <span class="btn btn-xs btn-danger">Required</span> Message Bot unique ID.
     *
     * @apiSampleRequest /whatsbotapi/v1/messagebots/id/1
     *
     * @apiSuccess {Boolean} status  Response status.
     * @apiSuccess {Object}  data    Message Bot.
     * @apiSuccess {String}  message Success message.
     *
     * @apiSuccessExample Success-Response:
     *     HTTP/1.1 200 OK
     *       {
     *            "status": true,
     *             "data": {
     *                       "id": "1",
     *                       "name": "test bot1",
     *                       "rel_type": "leads",
     *                       "reply_text": "hello from test bot 1",
     *                       "reply_type": "1",
     *                       "trigger": "testbot",
     *                       "bot_header": "",
     *                       "bot_footer": "",
     *                       "button1": "",
     *                       "button1_id": "",
     *                       "button2": "",
     *                       "button2_id": "",
     *                       "button3": "",
     *                       "button3_id": "",
     *                       "button_name": "",
     *                       "button_url": "",
     *                       "filename": null,
     *                       "addedfrom": "1",
     *                       "created_at": "2024-09-21 12:09:30",
     *                       "is_bot_active": "1",
     *                       "sending_count": "0"
     *       },
     *       "message": "Data retrived successfully"
     * }
     * @apiError {Boolean} status  Response status.
     * @apiError {String}  message Error message.
     *
     * @apiErrorExample Error-Response:
     *     HTTP/1.1 404 Not Found
     *     {
     *       "status": false,
     *       "message": "No data found"
     *     }
     */


    public function messagebots_get()
    {
        if (!empty($this->get()) && !in_array('id', array_keys($this->get()))) {
            $this->response(['message' => _l('something_went_wrong')], 500);
        }

        if (isset($this->get()['id']) && !is_numeric($this->get('id'))) {
            $this->response(['message' => _l('id_must_be_numeric')], 500);
        }

        $botID   = $this->get('id');
        $botData = $this->whatsbotapi_model->getTable('wtc_messagebots', $botID);

        $this->response($botData['response'], $botData['response_code']);
    }

    /**
     * @api {post} /whatsbotapi/v1/messagebots Create new message bot
     *
     * @apiVersion 1.0.0
     *
     * @apiName Create new message bot
     *
     * @apiGroup Message Bots
     *
     * @apiSampleRequest off
     *
     * @apiBody {String} name               <span class="btn btn-xs btn-danger">Required</span> Bot name
     * @apiBody {String} rel_type           <span class="btn btn-xs btn-danger">Required</span> Relation type
     * @apiBody {String} reply_text         <span class="btn btn-xs btn-danger">Required</span> Replay text
     * @apiBody {integer} reply_type        <span class="btn btn-xs btn-danger">Required</span> Replay type
     * @apiBody {String} trigger            <span class="btn btn-xs btn-danger">Required</span> Trigger keyword
     * @apiBody {String} [bot_header]       Header
     * @apiBody {String} [bot_footer]       Footer
     * @apiBody {String} [button1]          Button 1
     * @apiBody {String} [button1_id]       Button 1 id
     * @apiBody {String} [button2]          Button 2
     * @apiBody {String} [button2_id]       Button 2 id
     * @apiBody {String} [button3]          Button 3
     * @apiBody {String} [button3_id]       Button 3 id
     * @apiBody {String} [button_name]      Button name
     * @apiBody {String} [button_url]       Button url
     * @apiBody {String} [file_type]        File type
     * @apiBody {String} [file]             File
     *
     * @apiSuccess {Boolean} status  Response status.
     * @apiSuccess {String}  message Bot created successfully.
     *
     * @apiSuccessExample Success-Response:
     *     HTTP/1.1 200 OK
     *     {
     *         "status": true,
     *         "message": "Bot created successfully."
     *     }
     *
     * @apiError {Boolean} status  Response status.
     * @apiError {String}  message Error message.
     *
     * @apiErrorExample Error-Response:
     *     HTTP/1.1 400 Bad Request
     *     {
     *       "status": false,
     *       "message": "Something went wrong"
     *     }
     */

    public function messagebots_post()
    {
        $requiredData = [
            'name' => '',
            'rel_type' => '',
            'reply_text' => '',
            'reply_type' => '',
            'trigger' => ''
        ];

        $postData = $this->post();
        $postData = array_merge($requiredData, $postData);

        $this->load->library('form_validation');

        $this->form_validation->set_data($postData);

        $this->form_validation->set_rules('name', _l('bot_name'), 'required');
        $this->form_validation->set_rules('rel_type', _l('rel_type'), 'required|in_list[leads,contacts]');
        $this->form_validation->set_rules('reply_text', _l('reply_text'), 'required|max_length[1024]');
        $this->form_validation->set_rules('reply_type', _l('reply_type'), 'required|numeric|in_list[1,2,3,4]');
        $this->form_validation->set_rules('trigger', _l('trigger'), 'required');
        $this->form_validation->set_rules('bot_header', _l('bot_header'), 'max_length[60]');
        $this->form_validation->set_rules('bot_footer', _l('bot_footer'), 'max_length[60]');
        // Buttons validation
        $this->form_validation->set_rules('button1', _l('button1'), 'max_length[20]');
        $this->form_validation->set_rules('button1_id', _l('button1_id'), 'max_length[256]');
        $this->form_validation->set_rules('button2', _l('button2'), 'max_length[20]');
        $this->form_validation->set_rules('button2_id', _l('button2_id'), 'max_length[256]');
        $this->form_validation->set_rules('button3', _l('button3'), 'max_length[20]');
        $this->form_validation->set_rules('button3_id', _l('button3_id'), 'max_length[256]');

        $this->form_validation->set_rules('button_name', _l('button_name'), 'max_length[20]');
        $this->form_validation->set_rules('button_url', _l('button_url'), 'valid_url');
        // File validations
        $this->form_validation->set_rules('file_type', _l('file_type'), 'in_list[image,document]');

        if ($this->form_validation->run() === FALSE) {
            $this->response(['message' => strip_tags(validation_errors())], 422);
        }

        $res = $this->bots_model->saveBots($postData);

        if ($res['type'] == 'success') {
            wb_handle_whatsbot_upload($res['id']);
        }

        ($res['type'] == 'success') ? $this->response(['message' => $res['message']], 200) : $this->response(['message' => _l('something_went_wrong')], 500);
    }

    public function messagebots_delete()
    {
        if (empty($this->get()) && !in_array('id', array_keys($this->get()))) {
            $this->response(['message' => _l('something_went_wrong')], 500);
        }

        if (isset($this->get()['id']) && !is_numeric($this->get('id'))) {
            $this->response(['message' => _l('id_must_be_numeric')], 500);
        }

        $bot_id = $this->get('id');
        $res = $this->bots_model->deleteMessageBot('message', $bot_id);

        ($res['status']) ? $this->response(['message' => _l('bot_deleted_successfully')], 200) : $this->response(['message' => _l('no_record_found')], 500);
    }
}
